﻿// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

using System;
using System.Diagnostics.CodeAnalysis;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Microsoft.CodeAnalysis.CSharp;
using Microsoft.CodeAnalysis.CSharp.Formatting;
using Microsoft.CodeAnalysis.CSharp.Test.Utilities;
using Microsoft.CodeAnalysis.Editing;
using Microsoft.CodeAnalysis.Editor.UnitTests.CodeActions;
using Microsoft.CodeAnalysis.Editor.UnitTests.Utilities;
using Microsoft.CodeAnalysis.Formatting;
using Microsoft.CodeAnalysis.Indentation;
using Microsoft.CodeAnalysis.Shared.Extensions;
using Microsoft.CodeAnalysis.Test.Utilities;
using Microsoft.VisualStudio.Text.Editor.Commanding.Commands;
using Roslyn.Test.Utilities;
using Xunit;
using Xunit.Abstractions;

namespace Microsoft.CodeAnalysis.Editor.CSharp.UnitTests.Formatting;

[Trait(Traits.Feature, Traits.Features.Formatting)]
public sealed class FormattingEngineTests(ITestOutputHelper output) : CSharpFormattingEngineTestBase(output)
{
    private static OptionsCollection SmartIndentButDoNotFormatWhileTyping()
        => new(LanguageNames.CSharp)
        {
            { IndentationOptionsStorage.SmartIndent, FormattingOptions2.IndentStyle.Smart },
            { AutoFormattingOptionsStorage.FormatOnTyping, false },
            { AutoFormattingOptionsStorage.FormatOnCloseBrace, false },
        };

    [WpfFact, WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/539682")]
    public void FormatDocumentCommandHandler()
        => AssertFormatWithView("""
                class Program
                {
                    static void Main(string[] args)
                    {
                        int x;$$
                        int y;
                    }
                }
                """, """
                class Program
                {
                    static void Main(string[] args)
                    {
                        int x;$$
                int y;
                    }
                }
                """);

    [WpfFact, WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/539682")]
    public void FormatDocumentPasteCommandHandler()
        => AssertFormatWithPasteOrReturn("""
                class Program
                {
                    static void Main(string[] args)
                    {
                        int x;$$
                        int y;
                    }
                }
                """, """
                class Program
                {
                    static void Main(string[] args)
                    {
                        int x;$$
                int y;
                    }
                }
                """, allowDocumentChanges: true);

    [WpfFact, WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/547261")]
    public void FormatDocumentReadOnlyWorkspacePasteCommandHandler()
        => AssertFormatWithPasteOrReturn("""
                class Program
                {
                    static void Main(string[] args)
                    {
                        int x;$$
                int y;
                    }
                }
                """, """
                class Program
                {
                    static void Main(string[] args)
                    {
                        int x;$$
                int y;
                    }
                }
                """, allowDocumentChanges: false);

    [WpfFact, WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/912965")]
    public void DoNotFormatUsingStatementOnReturn()
        => AssertFormatWithPasteOrReturn("""
                class Program
                {
                    static void Main(string[] args)
                    {
                        using (null)
                                using (null)$$
                    }
                }
                """, """
                class Program
                {
                    static void Main(string[] args)
                    {
                        using (null)
                                using (null)$$
                    }
                }
                """, allowDocumentChanges: true, isPaste: false);

    [WpfFact, WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/912965")]
    public void FormatUsingStatementWhenTypingCloseParen()
        => AssertFormatAfterTypeChar("""
                class Program
                {
                    static void Main(string[] args)
                    {
                        using (null)
                                using (null)$$
                    }
                }
                """, """
                class Program
                {
                    static void Main(string[] args)
                    {
                        using (null)
                        using (null)
                    }
                }
                """);

    [WpfFact, WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/912965")]
    public void FormatNotUsingStatementOnReturn()
        => AssertFormatWithPasteOrReturn("""
                class Program
                {
                    static void Main(string[] args)
                    {
                        using (null)
                                for (;;)$$
                    }
                }
                """, """
                class Program
                {
                    static void Main(string[] args)
                    {
                        using (null)
                                for (;;)$$
                    }
                }
                """, allowDocumentChanges: true, isPaste: false);

    [WpfFact, WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/977133")]
    public void DoNotFormatRangeOrFormatTokenOnOpenBraceOnSameLine()
        => AssertFormatAfterTypeChar("""
                class C
                {
                    public void M()
                    {
                        if (true)        {$$
                    }
                }
                """, """
                class C
                {
                    public void M()
                    {
                        if (true)        {
                    }
                }
                """);

    [WpfFact, WorkItem("https://github.com/dotnet/roslyn/pull/14491")]
    public void DoNotFormatRangeButFormatTokenOnOpenBraceOnNextLine()
        => AssertFormatAfterTypeChar("""
                class C
                {
                    public void M()
                    {
                        if (true)
                            {$$
                    }
                }
                """, """
                class C
                {
                    public void M()
                    {
                        if (true)
                        {
                    }
                }
                """);

    [WpfFact, WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/1007071")]
    public void FormatPragmaWarningInbetweenDelegateDeclarationStatement()
        => AssertFormatAfterTypeChar("""
                using System;

                class Program
                {
                    static void Main(string[] args)
                    {
                        Func <bool> a = delegate ()
                #pragma warning disable CA0001
                        {
                            return true;
                        };$$
                    }
                }
                """, """
                using System;

                class Program
                {
                    static void Main(string[] args)
                    {
                        Func<bool> a = delegate ()
                #pragma warning disable CA0001
                        {
                            return true;
                        };
                    }
                }
                """);

    [WpfFact, WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/771761")]
    public void FormatHashRegion()
        => AssertFormatAfterTypeChar("""
                using System;

                class Program
                {
                    static void Main(string[] args)
                    {
                #region$$
                    }
                }
                """, """
                using System;

                class Program
                {
                    static void Main(string[] args)
                    {
                        #region
                    }
                }
                """);

    [WpfFact, WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/771761")]
    public void FormatHashEndRegion()
        => AssertFormatAfterTypeChar("""
                using System;

                class Program
                {
                    static void Main(string[] args)
                    {
                        #region
                #endregion$$
                    }
                }
                """, """
                using System;

                class Program
                {
                    static void Main(string[] args)
                    {
                        #region
                        #endregion
                    }
                }
                """);

    [WpfFact, WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/987373")]
    public async Task FormatSpansIndividuallyWithoutCollapsing()
    {
        var code = """
                class C
                {
                    public void M()
                    {
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        [|if(true){}|]
                        if(true){}
                        [|if(true){}|]
                    }
                }
                """;
        using var workspace = EditorTestWorkspace.CreateCSharp(code);
        var subjectDocument = workspace.Documents.Single();
        var spans = subjectDocument.SelectedSpans;

        var document = workspace.CurrentSolution.Projects.Single().Documents.Single();
        var syntaxRoot = await document.GetRequiredSyntaxRootAsync(CancellationToken.None);
        var options = CSharpSyntaxFormattingOptions.Default;
        var node = Formatter.Format(syntaxRoot, spans, workspace.Services.SolutionServices, options, rules: default, CancellationToken.None);
        Assert.Equal("""
                class C
                {
                    public void M()
                    {
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if (true) { }
                        if(true){}
                        if (true) { }
                    }
                }
                """, node.ToFullString());
    }

    [WpfFact, WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/1044118")]
    public void SemicolonInCommentOnLastLineDoesNotFormat()
        => AssertFormatAfterTypeChar("""
                using System;

                class Program
                {
                    static void Main(string[] args)
                        {
                        }
                }
                // ;$$
                """, """
                using System;

                class Program
                {
                    static void Main(string[] args)
                        {
                        }
                }
                // ;
                """);

    [WpfFact, WorkItem("https://github.com/dotnet/roslyn/issues/449")]
    [WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/1077103")]
    public void NoFormattingInsideSingleLineRegularComment_1()
        => AssertFormatAfterTypeChar("""
                class Program
                {
                                              //        {$$
                                       static void Main(int a, int b)
                    {

                    }
                }
                """, """
                class Program
                {
                                              //        {
                                       static void Main(int a, int b)
                    {

                    }
                }
                """);

    [WpfFact, WorkItem("https://github.com/dotnet/roslyn/issues/449")]
    [WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/1077103")]
    public void NoFormattingInsideSingleLineRegularComment_2()
        => AssertFormatAfterTypeChar("""
                class Program
                {
                                              //        {$$   
                                       static void Main(int a, int b)
                    {

                    }
                }
                """, """
                class Program
                {
                                              //        {   
                                       static void Main(int a, int b)
                    {

                    }
                }
                """);

    [WpfFact, WorkItem("https://github.com/dotnet/roslyn/issues/449")]
    [WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/1077103")]
    public void NoFormattingInsideMultiLineRegularComment_1()
        => AssertFormatAfterTypeChar("""
                class Program
                {
                    static void Main(int          a/*         {$$       */, int b)
                    {

                    }
                }
                """, """
                class Program
                {
                    static void Main(int          a/*         {       */, int b)
                    {

                    }
                }
                """);

    [WpfFact, WorkItem("https://github.com/dotnet/roslyn/issues/449")]
    [WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/1077103")]
    public void NoFormattingInsideMultiLineRegularComment_2()
        => AssertFormatAfterTypeChar("""
                class Program
                {
                    static void Main(int          a/*         {$$
                        */, int b)
                    {

                    }
                }
                """, """
                class Program
                {
                    static void Main(int          a/*         {
                        */, int b)
                    {

                    }
                }
                """);

    [WpfFact, WorkItem("https://github.com/dotnet/roslyn/issues/449")]
    [WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/1077103")]
    public void NoFormattingInsideMultiLineRegularComment_3()
        => AssertFormatAfterTypeChar("""
                class Program
                {
                    static void Main(int          a/*         {$$    
                        */, int b)
                    {

                    }
                }
                """, """
                class Program
                {
                    static void Main(int          a/*         {    
                        */, int b)
                    {

                    }
                }
                """);

    [WpfFact, WorkItem("https://github.com/dotnet/roslyn/issues/449")]
    [WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/1077103")]
    public void NoFormattingInsideSingleLineDocComment_1()
        => AssertFormatAfterTypeChar("""
                class Program
                {
                                              ///        {$$
                                       static void Main(int a, int b)
                    {

                    }
                }
                """, """
                class Program
                {
                                              ///        {
                                       static void Main(int a, int b)
                    {

                    }
                }
                """);

    [WpfFact, WorkItem("https://github.com/dotnet/roslyn/issues/449")]
    [WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/1077103")]
    public void NoFormattingInsideSingleLineDocComment_2()
        => AssertFormatAfterTypeChar("""
                class Program
                {
                                              ///        {$$   
                                       static void Main(int a, int b)
                    {

                    }
                }
                """, """
                class Program
                {
                                              ///        {   
                                       static void Main(int a, int b)
                    {

                    }
                }
                """);

    [WpfFact, WorkItem("https://github.com/dotnet/roslyn/issues/449")]
    [WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/1077103")]
    public void NoFormattingInsideMultiLineDocComment_1()
        => AssertFormatAfterTypeChar("""
                class Program
                {
                                              /**        {$$   **/
                                       static void Main(int a, int b)
                    {

                    }
                }
                """, """
                class Program
                {
                                              /**        {   **/
                                       static void Main(int a, int b)
                    {

                    }
                }
                """);

    [WpfFact, WorkItem("https://github.com/dotnet/roslyn/issues/449")]
    [WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/1077103")]
    public void NoFormattingInsideMultiLineDocComment_2()
        => AssertFormatAfterTypeChar("""
                class Program
                {
                                              /**        {$$   
                                **/
                                       static void Main(int a, int b)
                    {

                    }
                }
                """, """
                class Program
                {
                                              /**        {   
                                **/
                                       static void Main(int a, int b)
                    {

                    }
                }
                """);

    [WpfFact, WorkItem("https://github.com/dotnet/roslyn/issues/449")]
    [WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/1077103")]
    public void NoFormattingInsideMultiLineDocComment_3()
        => AssertFormatAfterTypeChar("""
                class Program
                {
                                              /**        {$$
                                **/
                                       static void Main(int a, int b)
                    {

                    }
                }
                """, """
                class Program
                {
                                              /**        {
                                **/
                                       static void Main(int a, int b)
                    {

                    }
                }
                """);

    [WpfFact, WorkItem("https://github.com/dotnet/roslyn/issues/449")]
    [WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/1077103")]
    public void NoFormattingInsideInactiveCode()
        => AssertFormatAfterTypeChar("""
                class Program
                {
                                                        #if false
                                    {$$
                            #endif

                    static void Main(string[] args)
                    {

                    }
                }
                """, """
                class Program
                {
                                                        #if false
                                    {
                            #endif

                    static void Main(string[] args)
                    {

                    }
                }
                """);

    [WpfFact, WorkItem("https://github.com/dotnet/roslyn/issues/449")]
    [WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/1077103")]
    public void NoFormattingInsideStringLiteral()
        => AssertFormatAfterTypeChar("""
                class Program
                {
                    static void Main(string[] args)
                    {
                        var asdas =     "{$$"        ;
                    }
                }
                """, """
                class Program
                {
                    static void Main(string[] args)
                    {
                        var asdas =     "{"        ;
                    }
                }
                """);

    [WpfFact, WorkItem("https://github.com/dotnet/roslyn/issues/449")]
    [WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/1077103")]
    public void NoFormattingInsideCharLiteral()
        => AssertFormatAfterTypeChar("""
                class Program
                {
                    static void Main(string[] args)
                    {
                        var asdas =     '{$$'        ;
                    }
                }
                """, """
                class Program
                {
                    static void Main(string[] args)
                    {
                        var asdas =     '{'        ;
                    }
                }
                """);

    [WpfFact, WorkItem("https://github.com/dotnet/roslyn/issues/449")]
    [WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/1077103")]
    public void NoFormattingInsideCommentsOfPreprocessorDirectives()
        => AssertFormatAfterTypeChar("""
                class Program
                {
                       #region
                        #endregion // a/*{$$*/    
                        static void Main(string[] args)
                    {

                    }
                }
                """, """
                class Program
                {
                       #region
                        #endregion // a/*{*/    
                        static void Main(string[] args)
                    {

                    }
                }
                """);

    [WpfFact, WorkItem("https://github.com/dotnet/roslyn/issues/464")]
    [WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/908729")]
    public void ColonInSwitchCase()
        => AssertFormatAfterTypeChar("""
                class Program
                {
                    static void Main(string[] args)
                    {
                        int f = 0;
                        switch(f)
                        {
                                 case     1     :$$    break;
                        }
                    }
                }
                """, """
                class Program
                {
                    static void Main(string[] args)
                    {
                        int f = 0;
                        switch(f)
                        {
                            case 1:    break;
                        }
                    }
                }
                """);

    [WpfFact, WorkItem("https://github.com/dotnet/roslyn/issues/464")]
    [WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/908729")]
    public void ColonInDefaultSwitchCase()
        => AssertFormatAfterTypeChar("""
                class Program
                {
                    static void Main(string[] args)
                    {
                        int f = 0;
                        switch(f)
                        {
                            case 1:    break;
                                    default    :$$     break;
                        }
                    }
                }
                """, """
                class Program
                {
                    static void Main(string[] args)
                    {
                        int f = 0;
                        switch(f)
                        {
                            case 1:    break;
                            default:     break;
                        }
                    }
                }
                """);

    [WpfFact, WorkItem("https://github.com/dotnet/roslyn/issues/9097")]
    public void ColonInPatternSwitchCase01()
        => AssertFormatAfterTypeChar("""
                class Program
                {
                    static void Main()
                    {
                        switch(f)
                        {
                                          case  int  i            :$$    break;
                        }
                    }
                }
                """, """
                class Program
                {
                    static void Main()
                    {
                        switch(f)
                        {
                            case int i:    break;
                        }
                    }
                }
                """);

    [WpfFact, WorkItem("https://github.com/dotnet/roslyn/issues/464")]
    [WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/908729")]
    public void ColonInLabeledStatement()
        => AssertFormatAfterTypeChar("""
                class Program
                {
                    static void Main(string[] args)
                    {
                            label1   :$$   int s = 0;
                    }
                }
                """, """
                class Program
                {
                    static void Main(string[] args)
                    {
                    label1: int s = 0;
                    }
                }
                """);

    [WpfFact, WorkItem("https://github.com/dotnet/roslyn/issues/464")]
    [WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/908729")]
    public void DoNotFormatColonInTargetAttribute()
        => AssertFormatAfterTypeChar("""
                using System;
                [method    :$$    C]
                class C : Attribute
                {
                }
                """, """
                using System;
                [method    :    C]
                class C : Attribute
                {
                }
                """);

    [WpfFact, WorkItem("https://github.com/dotnet/roslyn/issues/464")]
    [WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/908729")]
    public void DoNotFormatColonInBaseList()
        => AssertFormatAfterTypeChar("""
                class C   :$$   Attribute
                {
                }
                """, """
                class C   :   Attribute
                {
                }
                """);

    [WpfFact, WorkItem("https://github.com/dotnet/roslyn/issues/464")]
    [WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/908729")]
    public void DoNotFormatColonInThisConstructor()
        => AssertFormatAfterTypeChar("""
                class Goo
                {
                    Goo(int s)   :$$   this()
                    {
                    }

                    Goo()
                    {
                    }
                }
                """, """
                class Goo
                {
                    Goo(int s)   :   this()
                    {
                    }

                    Goo()
                    {
                    }
                }
                """);

    [WpfFact, WorkItem("https://github.com/dotnet/roslyn/issues/464")]
    [WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/908729")]
    public void DoNotFormatColonInConditionalOperator()
        => AssertFormatAfterTypeChar("""
                class Program
                {
                    static void Main(string[] args)
                    {
                        var vari = goo()     ?    true  :$$  false;
                    }
                }
                """, """
                class Program
                {
                    static void Main(string[] args)
                    {
                        var vari = goo()     ?    true  :  false;
                    }
                }
                """);

    [WpfFact, WorkItem("https://github.com/dotnet/roslyn/issues/464")]
    [WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/908729")]
    public void DoNotFormatColonInArgument()
        => AssertFormatAfterTypeChar("""
                class Program
                {
                    static void Main(string[] args)
                    {
                        Main(args  :$$  args);
                    }
                }
                """, """
                class Program
                {
                    static void Main(string[] args)
                    {
                        Main(args  :  args);
                    }
                }
                """);

    [WpfFact, WorkItem("https://github.com/dotnet/roslyn/issues/464")]
    [WorkItem("http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/908729")]
    public void DoNotFormatColonInTypeParameter()
        => AssertFormatAfterTypeChar("""
                class Program<T>
                {
                    class C1<U>
                        where   T  :$$  U
                    {

                    }
                }
                """, """
                class Program<T>
                {
                    class C1<U>
                        where   T  :  U
                    {

                    }
                }
                """);

    [WpfFact, WorkItem("https://github.com/dotnet/roslyn/issues/2224")]
    public void DoNotSmartFormatBracesOnSmartIndentNone()
    {
        var globalOptions = new OptionsCollection(LanguageNames.CSharp)
        {
            { IndentationOptionsStorage.SmartIndent, FormattingOptions2.IndentStyle.None }
        };
        AssertFormatAfterTypeChar("""
                class Program<T>
                {
                    class C1<U>
                {$$
                }
                """, """
                class Program<T>
                {
                    class C1<U>
                {
                }
                """, globalOptions);
    }

    [WpfFact]
    [Trait(Traits.Feature, Traits.Features.SmartTokenFormatting)]
    public void StillAutoIndentCloseBraceWhenFormatOnCloseBraceIsOff()
    {
        var globalOptions = new OptionsCollection(LanguageNames.CSharp)
        {
            { AutoFormattingOptionsStorage.FormatOnCloseBrace, false }
        };

        AssertFormatAfterTypeChar("""
                namespace N
                {
                    class C
                    {
                             // improperly indented code
                             int x = 10;
                        }$$
                }
                """, """
                namespace N
                {
                    class C
                    {
                             // improperly indented code
                             int x = 10;
                    }
                }
                """, globalOptions);
    }

    [WpfFact]
    [Trait(Traits.Feature, Traits.Features.SmartTokenFormatting)]
    public void AutoIndentCloseBraceWhenFormatOnTypingIsOff()
    {
        var globalOptions = new OptionsCollection(LanguageNames.CSharp)
        {
            { AutoFormattingOptionsStorage.FormatOnTyping, false }
        };

        AssertFormatAfterTypeChar("""
                namespace N
                {
                    class C
                    {
                             // improperly indented code
                             int x = 10;
                        }$$
                }
                """, """
                namespace N
                {
                    class C
                    {
                             // improperly indented code
                             int x = 10;
                    }
                }
                """, globalOptions);
    }

    [WpfFact, Trait(Traits.Feature, Traits.Features.SmartTokenFormatting)]
    [WorkItem("https://github.com/dotnet/roslyn/issues/5873")]
    public void KeepTabsInCommentsWhenFormattingIsOff()
    {
        var globalOptions = new OptionsCollection(LanguageNames.CSharp)
        {
            { AutoFormattingOptionsStorage.FormatOnTyping, false }
        };

        AssertFormatAfterTypeChar("""
                class Program
                {
                    static void Main()
                    {
                        return;		/* Comment preceded by tabs */		// This one too
                        }$$
                }
                """, """
                class Program
                {
                    static void Main()
                    {
                        return;		/* Comment preceded by tabs */		// This one too
                    }
                }
                """, globalOptions);
    }

    [WpfFact, Trait(Traits.Feature, Traits.Features.SmartTokenFormatting)]
    [WorkItem("https://github.com/dotnet/roslyn/issues/5873")]
    public void DoNotKeepTabsInCommentsWhenFormattingIsOn()
        => AssertFormatAfterTypeChar("""
                class Program
                {
                    static void Main()
                    {
                        return;		/* Comment preceded by tabs */		// This one too
                        }$$
                }
                """, """
                class Program
                {
                    static void Main()
                    {
                        return;     /* Comment preceded by tabs */        // This one too
                    }
                }
                """);

    [WpfFact]
    [Trait(Traits.Feature, Traits.Features.SmartTokenFormatting)]
    public void DoNotFormatStatementIfSemicolonOptionIsOff()
    {
        var globalOptions = new OptionsCollection(LanguageNames.CSharp)
        {
            { AutoFormattingOptionsStorage.FormatOnSemicolon, false }
        };

        AssertFormatAfterTypeChar("""
                namespace N
                {
                    class C
                    {
                        int x   =   10     ;$$
                    }
                }
                """, """
                namespace N
                {
                    class C
                    {
                        int x   =   10     ;
                    }
                }
                """, globalOptions);
    }

    [WpfFact]
    [Trait(Traits.Feature, Traits.Features.SmartTokenFormatting)]
    public void DoNotFormatStatementIfTypingOptionIsOff()
    {
        var globalOptions = new OptionsCollection(LanguageNames.CSharp)
        {
            { AutoFormattingOptionsStorage.FormatOnTyping, false }
        };

        AssertFormatAfterTypeChar("""
                namespace N
                {
                    class C
                    {
                        int x   =   10     ;$$
                    }
                }
                """, """
                namespace N
                {
                    class C
                    {
                        int x   =   10     ;
                    }
                }
                """, globalOptions);
    }

    [WpfFact, WorkItem("https://github.com/dotnet/roslyn/issues/4435")]
    [Trait(Traits.Feature, Traits.Features.SmartTokenFormatting)]
    public void OpenCurlyNotFormattedIfNotAtStartOfLine()
        => AssertFormatAfterTypeChar("""
                class C
                {
                    public  int     P   {$$
                }
                """, """
                class C
                {
                    public  int     P   {
                }
                """);

    [WpfFact, WorkItem("https://github.com/dotnet/roslyn/issues/4435")]
    [Trait(Traits.Feature, Traits.Features.SmartTokenFormatting)]
    public void OpenCurlyFormattedIfAtStartOfLine()
        => AssertFormatAfterTypeChar("""
                class C
                {
                    public  int     P
                        {$$
                }
                """, """
                class C
                {
                    public  int     P
                    {
                }
                """);

    [WpfFact]
    public void DoNotFormatIncompleteBlockOnSingleLineIfNotTypingCloseCurly1()
        => AssertFormatAfterTypeChar("""
                namespace ConsoleApplication1
                {
                    class Program
                    {
                        static bool Property
                        {
                            get { return true;$$
                    }
                }
                """, """
                namespace ConsoleApplication1
                {
                    class Program
                    {
                        static bool Property
                        {
                            get { return true;
                    }
                }
                """);

    [WpfFact]
    public void DoNotFormatIncompleteBlockOnSingleLineIfNotTypingCloseCurly2()
        => AssertFormatAfterTypeChar("""
                namespace ConsoleApplication1
                {
                    class Program
                    {
                        static bool Property { get { return true;$$
                    }
                }
                """, """
                namespace ConsoleApplication1
                {
                    class Program
                    {
                        static bool Property { get { return true;
                    }
                }
                """);

    [WpfFact]
    public void DoNotFormatIncompleteBlockOnSingleLineIfNotTypingCloseCurly3()
        => AssertFormatAfterTypeChar("""
                namespace ConsoleApplication1
                {
                    class Program
                    {
                        static bool Property { get;$$
                    }
                }
                """, """
                namespace ConsoleApplication1
                {
                    class Program
                    {
                        static bool Property { get;
                    }
                }
                """);

    [WpfFact]
    public void DoNotFormatCompleteBlockOnSingleLineIfTypingCloseCurly1()
        => AssertFormatAfterTypeChar("""
                namespace ConsoleApplication1
                {
                    class Program
                    {
                        static bool Property
                        {
                            get { return true; }$$
                }
                """, """
                namespace ConsoleApplication1
                {
                    class Program
                    {
                        static bool Property
                        {
                            get { return true; }
                }
                """);

    [WpfFact]
    public void DoNotFormatCompleteBlockOnSingleLineIfTypingCloseCurly2()
        => AssertFormatAfterTypeChar("""
                namespace ConsoleApplication1
                {
                    class Program
                    {
                        static bool Property { get { return true; }$$
                }
                """, """
                namespace ConsoleApplication1
                